/*
 * Copyright (C) Jan 2006 Mellanox Technologies Ltd. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 *  utils.h - Miscellaneous general routines
 *
 *  Version: $Id: utils.h 2752 2006-01-19 14:40:17Z mst $
 *
 */

#ifndef UTILS_H
#define UTILS_H

#include <stdarg.h>

#include <list>
#include <vector>
#include <string>

#include "compatibility.h"

// don't forget to delete char* ptr returned from this!!!
char *print(const char *format, ...)
#ifdef __GNUC__
    __attribute__ ((format (printf, 1, 2)))
#endif
    ;

// don't forget to delete char* ptr returned from this!!!
char *vprint(const char *format, va_list args);

//-------------------------------------------------------
//------------------------------ string parsing utilities
//-------------------------------------------------------
//
////////////////////////////////////////////////////////////////////////
std::string              next_token(std::string& str,
                                    const char *delimiters = " \t\r\n");
std::list<std::string>   splitl(const std::string& s,
                                const char *delimiters, bool allow_empty=false);
std::vector<std::string> splitv(const std::string& s,
                                const char *delimiters, bool allow_empty=false);
template <class Seq> std::string join(Seq& l, const char *sep)
{
    int         size = l.size();
    int         cnt  = 0;
    std::string rc;
    for (typename Seq::iterator i = l.begin(); i != l.end(); ++i)
    {
        rc += *i;
        if (++cnt < size)
            rc += sep;
    }
    return rc;
} // join


// See item 7 from "Effective STL" by Scott Meyers
struct DeleteObject
{
    template<typename T> void operator()(const T* ptr) const
        { delete ptr; }
};


//-------------------------------------------
//------------------------------ ErrMsg class
//-------------------------------------------
//
////////////////////////////////////////////////////////////////////////
class ErrMsg
{
public:
    ErrMsg() : _err(0)       {                           }
    ~ErrMsg()                { err_clear();              }
    const char *err() const  { return _err;              }
    void       err_clear()   { delete [] _err; _err = 0; }

protected:
    bool errmsg(const char *format, ...)
#ifdef __GNUC__
        __attribute__ ((format (printf, 2, 3)))
#endif
        ;

private:
    char       *_err;
};



#ifndef NO_REGEX

#include <regex.h>
//-------------------------------------------
//------------------------------ Regexp class
//-------------------------------------------
//
///////////////////////////////////////////////////////////////////////
class Regexp
{
public:
    Regexp()
        : _err(0), _valid(false), _compiled(false)
    {
    }
    Regexp(const char *str_re, const int flag = 0)
        : _err(0), _valid(false), _compiled(false)
    {
        compile(str_re, flag | REG_EXTENDED);
    }
    ~Regexp();

    bool       compile(const char *str_re, const int flags = 0);
    const char *err() const  { return _err;              }
    void       err_clear()   { delete [] _err; _err = 0; }

    bool                       match(const std::string& s);
    unsigned                   nmatches() const { return _matches.size(); }
    std::string                cap(const unsigned n);
    template <class Cont> Cont split(const std::string& s);
    std::list<std::string>     splitl(const std::string& s);
    std::vector<std::string>   splitv(const std::string& s);

private:
    char                       *_err;
    bool                       _valid;
    bool                       _compiled;
    regex_t                    _re;
    std::vector<std::string>   _matches;
};

#endif //NO_REGEX

/////////////////////////////////////////////////
//
// Endianess issues
//
/////////////////////////////////////////////////

std::vector<u_int8_t>&  operator<<(std::vector<u_int8_t>& v1, u_int8_t  b);
std::vector<u_int8_t>&  operator<<(std::vector<u_int8_t>& v1, u_int32_t w);
std::vector<u_int8_t>&  operator<<(std::vector<u_int8_t>& v1, std::vector<u_int8_t>&  v2);
std::vector<u_int8_t>&  operator<<(std::vector<u_int8_t>& v1, std::vector<u_int32_t>& v2);


#endif
